#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include "traverse.hh"
#include "board.hh"
#include "tile.hh"
#include <stdlib.h>

Moment Traversal::flash_on_delay(0, 400000);
Moment Traversal::flash_off_delay(0, 150000);


Traversal::Traversal(Board *b)
  : _board(b), _game(b->game()), _on(false),
    _cursor(NULL), _alarm(this)
{
  _game->add_hook(this);
  layout_hook(_game);
}

Traversal::~Traversal()
{
  _game->remove_hook(this);
}


void
Traversal::show(bool state)
{
  if (!_on || _state == state) return;
  _state = state;
  
  Tile *t = cursor();
  if (_state)
    _board->light(t);
  else if (_board->selected() != t)
    _board->unlight(t);
  _board->flush();
  
  Moment when = Moment::now();
  if (state)
    when += flash_on_delay;
  else
    when += flash_off_delay;
  _alarm.schedule(when);
}

void
Traversal::set_cursor(Tile *cursor)
{
  if (_on && _cursor != cursor) show(false);
  _cursor = cursor;
}

void
Traversal::turn_on(int r, int c, int l)
{
  if (_on) show(false);
  assert(r || c || l);
  _cursor = _game->grid(r, c, l);
  _on = true;
  show(true);
}


void
Traversal::turn_on(Tile *t)
{
  if (_on) show(false);
  assert(t);
  _cursor = t;
  _on = true;
  show(true);
}


void
Traversal::turn_off()
{
  if (_on) 
  {
    show(false);
    _on = false;
    for (int i = 0; i < _hint_on.size(); i++) {
      _board->set_tile_flag(_hint_on[i], Board::fKeepLit, false);
      _board->unlight(_hint_on[i]);
    }  
  }
}


void
Traversal::clear()
{
  turn_off();
  if (_cursor)
  {
    _horizontal_pos = _cursor->col();
    _vertical_pos = _cursor->row();
    _cursor = NULL;
  } 
  _hint_on.clear();
}


typedef Tile *pTile; 

int rowcomp(const pTile &t1, const pTile &t2)
{
  return (t1->row() > t2->row()) - (t1->row() < t2->row());
}    

int colcomp(const pTile &t1, const pTile &t2)
{
  return (t1->col() > t2->col()) - (t1->col() < t2->col());
}    


void
Traversal::create_horizontal()
{
  /* Horizontal traversal: What a massive pain in the ass traversal has been!
     Here's the problem. I seem to have an internal "algorithm" for what the
     traversal pattern should be, but it's based on information from the 2D
     board -- hard to code -- and even then, I'm not sure how to formalize
     it!! So the problem is to approximate this, using only relatively local
     information. To be "as right as possible" without ever being DEAD WRONG.
     (Being dead wrong some of the time is worse than being partially right
     all of the time, I assert from experience.)
     
     So here is the idea. Consider groups of 2 rows at a time, row r and row
     r + 1. Sort these together. The End.

     A slight improvement: Assume all boards are vertically symmetrical, and
     don't move indiscriminately over the axis of symmetry. This solves this
     problem:

        XX XX			If X and Y sort together, then Z and Q
      YYXX XXYY			should also sort together. But the presence
      YY     YYMM____axis	of M screwed this up: we had 3 groups --
      ZZ     ZZMM		X+Y, M+Z, Q.
      ZZQQ QQZZ
        QQ QQ			With axis consideration, this won't happen:
				we won't group M+Z because M is above the
				axis and Z is on it.
  */
  
  _horizontal.clear();
  for (int i=0; i<_game->ntiles(); i++)
    if (_game->tile(i)->real() && _game->tile(i)->open())
      _horizontal.push_back(_game->tile(i));
  _horizontal.sort(rowcomp);   
  _horizontal_index1 = 0;
  _horizontal_index2 = 0;
      
  int i1, i2 = 0;
  while (i2 < _horizontal.size() && 
         _horizontal[i2]->row() < _horizontal_symmetry_axis)
  {
    i1 = i2;
    while (i2 < _horizontal.size() && 
           _horizontal[i2]->row() <= _horizontal[i1]->row() + 1)
      i2++;
    if (_horizontal[i1]->row() <= _vertical_pos)
    { 
      _horizontal_index1 = _horizontal[i2-1]->row()>=_vertical_pos?i1:i2;
      _horizontal_index2 = i2;
    } 
    _horizontal.sort(colcomp, i1, i2);
  }
    
  int bound = i2;
  i1 = _horizontal.size() - 1;
  while (i1 >= bound)
  {  
    i2 = i1;
    while (i1 >= bound && 
           _horizontal[i1]->row() >= _horizontal[i2]->row() - 1)
      i1--;
    if (_horizontal_index2 <= i2 && _horizontal[i1+1]->row() <= _vertical_pos) 
    { 
      _horizontal_index1 = _horizontal[i2]->row()>=_vertical_pos?i1+1:i2+1;
      _horizontal_index2 = i2;
    } 
    _horizontal.sort(colcomp, i1+1, i2+1);
  }

}


void
Traversal::next_horiz(bool right)
{
  if (!_horizontal.size())
    create_horizontal();
  int horiz_count = _horizontal.size();
  if (!horiz_count) return;
  
  int i;
  if (_cursor)
  {
    for (i = 0; i < horiz_count; i++)
      if (_horizontal[i] == _cursor)
	break;
  }
  else
  {
    int i1 = _horizontal_index1;
    int i2 = _horizontal_index2;
    if (i1 == i2 || _horizontal[i1]->col() > _horizontal_pos)
    {  
      i = (i1 + horiz_count - 1 + right) % horiz_count;
    }   
    else
    {
      do i2--;
      while (_horizontal[i2]->col() > _horizontal_pos);
      if (_horizontal[i2]->col() == _horizontal_pos)
      {  
        i = i2;
      }   
      else
      {  
        i = (i2 + right) % horiz_count;
      }   
    }  
  }
  
  int delta = right ? 1 : horiz_count - 1;
  if (!_on && (!_cursor || _cursor != _board->selected()))
  {
    i = (i + horiz_count - delta) % horiz_count;
  }  
  _on = true;  
  for (int c = 0; c < horiz_count; c++) {
    i = (i + delta) % horiz_count;
    if (true) {
      turn_on(_horizontal[i]);
      return;
    }
  }

}


void
Traversal::create_vertical()
{
  /* merge columns "on the flow", such that the column will always change
     by at least two when pressing up or down.
  */   

  _vertical.clear();
  for (int i=0; i<_game->ntiles(); i++)
    if (_game->tile(i)->real() && _game->tile(i)->open())
      _vertical.push_back(_game->tile(i));
  _vertical.sort(colcomp);   
  _vertical_index2 = 0;
  _vertical_index1 = 0;
      
  int i1 = 0, i2 = 0;
  while (i2 < _vertical.size() &&
         _vertical[i2]->col() < _vertical_symmetry_axis)
  {
    i1 = i2;
    long collision = 3 << _vertical[i1]->row();
    int i3;
    for (i3=i1+1; i3<_vertical.size(); i3++)
    { 
      if (_vertical[i3]->col() > _vertical[i3-1]->col()) i2 = i3;
      if (_vertical[i2]->col() > _vertical_symmetry_axis) break; 
      long j = 3 << _vertical[i3]->row();
      if (j & collision) break;
      collision |= j;
    }
    if (i3 == _vertical.size()) i2 = i3;
    if (_vertical[i1]->col() <= _horizontal_pos)
    { 
      _vertical_index1 = _vertical[i2-1]->col()>=_horizontal_pos?i1:i2;
      _vertical_index2 = i2;
    }
    _vertical.sort(rowcomp, i1, i2);
  }  	 
     		     
  int bound = i2, i0 = i1;
  i1 = i2 = _vertical.size() - 1;
  while (i1 >= bound)
  {  
    i2 = i1;
    long collision = 3 << _vertical[i1]->row();
    int i3;
    for (i3=i1-1; i3>=bound; i3--)
    {
      if (_vertical[i3]->col() < _vertical[i3+1]->col()) i1 = i3;
      long j = 3 << _vertical[i3]->row();
      if (j & collision) break;
      collision |= j;
    }
    if (i3<bound) i1 = i3;
    if (_vertical_index2 <= i2 && _vertical[i1+1]->col() <= _horizontal_pos) 
    { 
      _vertical_index1 = _vertical[i2]->col()>=_horizontal_pos?i1+1:i2+1;
      _vertical_index2 = i2;
    } 
    _vertical.sort(rowcomp, i1+1, i2+1);
  }

  if (i0 <= i1 && i1 < i2)
  {
    int i;
    long collision1, collision2;
    for (i=i0,collision1=0; i<=i1; i++) 
      collision1 |= 3 << _vertical[i]->row();
    for (i=i1+1,collision2=0; i<=i2; i++) 
      collision2 |= 3 << _vertical[i]->row();
    if (!(collision1 & collision2)) _vertical.sort(rowcomp, i0, i2+1);
  }
}


void
Traversal::next_vert(bool down)
{
  if (!_vertical.size())
    create_vertical();
  int vert_count = _vertical.size();
  if (!vert_count) return;
  
  int i;
  if (_cursor)
  {
    for (i = 0; i < vert_count; i++)
      if (_vertical[i] == _cursor)
	break;
  }
  else
  {
    int i1 = _vertical_index1;
    int i2 = _vertical_index2;
    if (i1 == i2 || _vertical[i1]->row() > _vertical_pos)
    {  
      i = (i1 + vert_count - 1 + down) % vert_count;
    }   
    else
    {
      do i2--;
      while (_vertical[i2]->row() > _vertical_pos);
      if (_vertical[i2]->row() == _vertical_pos)
      {  
        i = i2;
      }   
      else
      {  
        i = (i2 + down) % vert_count;
      }   
    }  
  }
  
  int delta = down ? 1 : vert_count - 1;
  if (!_on && (!_cursor || _cursor != _board->selected()))
  {
    i = (i + vert_count - delta) % vert_count;
  }
  _on = true;
  for (int c = 0; c < vert_count; c++) {
    i = (i + delta) % vert_count;
    if (true) {
      turn_on(_vertical[i]);
      return;
    }
  }
}


void
Traversal::alarm()
{
  show(!_state);
}


void
Traversal::layout_hook(Game *g)
{
  if (g->ntiles())
  {
    int c1=1000, c2=0;
    for (int i=0; i<g->ntiles(); i++)
    {
      int c = g->tiles()[i]->row();
      if (c < c1) c1 = c;
      if (c > c2) c2 = c;
    }
    _horizontal_symmetry_axis = (c1 + c2) / 2;
    c1=1000, c2=0;
    for (int i=0; i<g->ntiles(); i++)
    {
      int c = g->tiles()[i]->col();
      if (c < c1) c1 = c;
      if (c > c2) c2 = c;
    }
    _vertical_symmetry_axis = (c1 + c2) / 2;
  }     				 
}

void
Traversal::start_hook(Game *)
{
  turn_off();
  _horizontal.clear();
  _vertical.clear();
  _hint_on.clear();
  _cursor = NULL;
  _horizontal_pos = -1;
  _vertical_pos = -1;
}

void
Traversal::add_tile_hook(Game *, Tile *)
{
  turn_off();
  _horizontal.clear();
  _vertical.clear();
  _hint_on.clear();
}

void
Traversal::remove_tile_hook(Game *, Tile *)
{
  turn_off();
  _horizontal.clear();
  _vertical.clear();
  _hint_on.clear();
}
